<?php
/*
 * @info Платёжный модуль Paygine для JoomShopping
 * @package JoomShopping for Joomla!
 * @subpackage payment
 * tested on Joomla v. 4.2.5 and JoomShopping v. 5.1.1
*/

defined('_JEXEC') or die('Restricted access');

class pm_paygine extends PaymentRoot {
	
	const EXT_NAME = 'paygine';
	const PARAMS_FIELDS = [
			'paygine_sector_id',
			'paygine_password',
			//'paygine_testmode', // test mode is on by default in the template
			//'paygine_tax', // default tax = 6
			'paygine_payment_type',
			'paygine_hash_algo',
			'transaction_end_status',
			'transaction_pending_status',
			'transaction_failed_status'
		];
	const HASH_ALGO_LIST = [
		'md5' => 'MD5',
		'sha256' => 'SHA256'
	];
	const PAYMENT_TYPES_PATHS = [
		'acquiring' => '/webapi/Purchase',
		'acquiring_two_stage' => '/webapi/Authorize',
		'svkb' => '/webapi/custom/svkb/PurchaseWithInstallment',
		'svkb_two_stage' => '/webapi/custom/svkb/AuthorizeWithInstallment',
		'sbp' => '/webapi/PurchaseSBP',
		//'loan' => '/webapi/custom/unicheckout/PurchaseWithLoanManager'
	];
	const DEFAULT_PAYMENT_PATH = '/webapi/Purchase';
	const FP_NAME_MAX_LENGTH = 128;
	const SHOP_CART_MAX_LENGTH = 2000;
	private $sector_id;
	private $password;
	private $testmode;
	private $payment_type;
	private $tax;
	private $hash_algo;
	
	function __construct()
	{
		$pm_method = \JSFactory::getTable('paymentMethod');
		$params = $pm_method->getConfigsForClassName(get_class($this));
		foreach($params as $name => $value){
			$short_name = str_replace(self::EXT_NAME . '_', '', $name);
			$this->{$short_name} = $value;
		}
	}
	function loadLanguageFile() {
		$lang = \JFactory::getApplication()->getLanguage();
		$lang->load('com_paygine', JPATH_ADMINISTRATOR);
	}

	function showPaymentForm($params, $pmconfigs) {
		include(dirname(__FILE__) . '/paymentform.php');
	}

	function showAdminFormParams($params) {
		$this->loadLanguageFile();
		
		foreach (self::PARAMS_FIELDS as $field) {
			$params[$field] = $params[$field] ?? '';
		}

		$orders = JModelLegacy::getInstance('orders', 'JshoppingModel');
		$payment_types_list = [
			'acquiring' => \JText::_('JSHOP_CFG_PAYGINE_ACQUIRING'),
			'acquiring_two_stage' => \JText::_('JSHOP_CFG_PAYGINE_ACQUIRING_TWO_STAGE'),
			'svkb' => \JText::_('JSHOP_CFG_PAYGINE_SVKB'),
			'svkb_two_stage' => \JText::_('JSHOP_CFG_PAYGINE_SVKB_TWO_STAGE'),
			'sbp' => \JText::_('JSHOP_CFG_PAYGINE_SBP'),
		];
		$tax_list = [
			1 => \JText::_('JSHOP_CFG_PAYGINE_TAX_1'),
			2 => \JText::_('JSHOP_CFG_PAYGINE_TAX_2'),
			3 => \JText::_('JSHOP_CFG_PAYGINE_TAX_3'),
			4 => \JText::_('JSHOP_CFG_PAYGINE_TAX_4'),
			5 => \JText::_('JSHOP_CFG_PAYGINE_TAX_5'),
			6 => \JText::_('JSHOP_CFG_PAYGINE_TAX_6')
		];
		
		include(dirname(__FILE__) . '/adminparamsform.php');
	}

	function checkTransaction($pmconfigs, $order, $act) {

		$this->loadLanguageFile();
		$request = \JFactory::getApplication()->input->getArray();

		try {
			$id = !empty($request['id']) ? (int) $request['id'] : null;
			$operation = !empty($request['operation']) ? (int) $request['operation'] : null;
			$data = [
				'sector' => $this->sector_id,
				'id' => $id,
				'operation' => $operation,
			];
			$this->sign_data($data, true);

			$repeat = 3;
			while ($repeat) {
				$repeat--;
				// pause because of possible background processing in the Paygine
				sleep(2);
				$xml = $this->sendRequest($this->getUrl() . '/webapi/Operation', $data);
				if (!$xml)
					throw new Exception("Empty data");
				$xml = simplexml_load_string($xml);
				if (!$xml)
					throw new Exception("Non valid XML was received");
				$response = json_decode(json_encode($xml), true);
				if (!$response)
					throw new Exception("Non valid XML was received");

				// check server signature
				$xml_string = self::xml_array_to_string($response);
				$signature = $this->generate_sign($xml_string, true);
				if ($signature !== $response['signature'])
					throw new Exception("Invalid signature");

				// check order state
				if (($response['type'] !== 'PURCHASE' && $response['type'] !== 'PURCHASE_BY_QR' && $response['type'] !== 'AUTHORIZE') || $response['state'] !== 'APPROVED')
					continue;

				return [1, ''];
			}
			
			return [0, \JText::_('JSHOP_CFG_PAYGINE_ORDER_NOT_PAID')];

		} catch (Exception $ex) {
			return [0, $ex->getMessage()];
		}
	}

	function showEndForm($pmconfigs, $order) {
		switch (strtolower($order->currency_code_iso)) {
			case 'usd':
				$currency = 840;
				break;
			case 'eur':
				$currency = 978;
				break;
			default:
				$currency = 643;
				break;
		}
		
		$order_amount = self::centify($order->order_total);

		$cart = JSFactory::getModel('cart', 'jshop');
		if (method_exists($cart, 'init')) {
			$cart->init('cart', 1);
		} else {
			$cart->load('cart');
		}
		
		$TAX = (isset($this->tax) && $this->tax > 0 && $this->tax <= 6) ? $this->tax : 6;
		$fiscalPositions = '';
		$fiscalAmount = 0;
		$shop_cart = [];
		$sc_key = 0;
		
		foreach ($cart->products as $product) {
			$product_quantity = (int) $product['quantity'];
			$product_price = self::centify($product['price']);
			$fiscalPositions .= $product_quantity . ';';
			$shop_cart[$sc_key]['quantityGoods'] = $product_quantity;
			$fiscalPositions .= $product_price . ';';
			$shop_cart[$sc_key]['goodCost'] = round($product['price'], 2);
			$fiscalPositions .= $TAX . ';';
			$fiscalPositions .= str_replace([';', '|'], '', mb_substr($product['product_name'], 0, self::FP_NAME_MAX_LENGTH)) . '|';
			$shop_cart[$sc_key]['name'] = $product['product_name'];
			$fiscalAmount += $product_quantity * $product_price;
			$sc_key++;
		}
		
		if ($order->order_shipping > 0) {
			$fiscalPositions .= '1;';
			$shop_cart[$sc_key]['quantityGoods'] = 1;
			$shipping_price = self::centify($order->order_shipping);
			$fiscalPositions .= $shipping_price . ';';
			$shop_cart[$sc_key]['goodCost'] = round($order->order_shipping, 2);
			$fiscalPositions .= $TAX . ';';
			$fiscalPositions .= \JText::_('JSHOP_SHIPPING') . '|';
			$shop_cart[$sc_key]['name'] = \JText::_('JSHOP_SHIPPING');
			$fiscalAmount += $shipping_price;
		}
		
		$fiscalDiff = abs($fiscalAmount - $order_amount);
		if ($fiscalDiff){
			$fiscalPositions .= '1' . ';';
			$fiscalPositions .= $fiscalDiff . ';';
			$fiscalPositions .= '6;';
			$fiscalPositions .= \JText::_('JSHOP_DISCOUNT') . ';';
			$fiscalPositions .= '14' . '|';
		}
		
		$fiscalPositions = substr($fiscalPositions, 0, -1);
		
		$signature = $this->generate_sign($this->sector_id . $order_amount . $currency, true);
		
		$data = [
			'sector' => $this->sector_id,
			'reference' => $order->order_id,
			'fiscal_positions' => $fiscalPositions,
			'amount' => $order_amount,
			'description' => sprintf(\JText::_('JSHOP_PAYMENT_NUMBER'), $order->order_number),
			'email' => $order->email,
			'currency' => $currency,
			'mode' => 1,
			'url' => JURI::root() . 'index.php?option=com_jshopping&controller=checkout&task=step7&act=return&js_paymentclass=pm_paygine',
			'signature' => $signature
		];
		error_log(var_export($data, true));

		$paygine_id = $this->sendRequest($this->getUrl() . '/webapi/Register', $data);
		
		if (intval($paygine_id) == 0) {
			$xml = simplexml_load_string($paygine_id);
			$response = json_decode(json_encode($xml), true);
			?>
			<html>
				<head>
					<meta http-equiv="content-type" content="text/html; charset=utf-8" />
				</head>
				<body>
					<p><?php echo $response['description']; ?></p>
					<p><a href="javascript:history.back();">«« <?php echo \JText::_('JSHOP_BACK_TO_SHOP')?></a></p>
				</body>
			</html>
			<?php
			die();
		}
		
		$payment_path = self::PAYMENT_TYPES_PATHS[$this->payment_type] ?? self::DEFAULT_PAYMENT_PATH;
		
		$post_data = [
			'sector' => $this->sector_id,
			'id' => $paygine_id,
		];
		
		$shop_cart_encoded = '';
		if($shop_cart && ($this->payment_type === 'svkb' || $this->payment_type === 'svkb_two_stage')) {
			$shop_cart_encoded = base64_encode(json_encode($shop_cart, JSON_UNESCAPED_UNICODE));
			if(strlen($shop_cart_encoded) > self::SHOP_CART_MAX_LENGTH)
				$shop_cart_encoded = '';
			$post_data['shop_cart'] = $shop_cart_encoded;
		}
		
		$post_data['signature'] = $this->generate_sign($this->sector_id . $paygine_id . $shop_cart_encoded, true);
		
		?>
		<html>
		<head>
			<meta http-equiv="content-type" content="text/html; charset=utf-8" />
		</head>
		<body>
			<form id="paymentform" accept-charset="utf8" action="<?php echo $this->getUrl() . $payment_path; ?>" method="post">
				<?php foreach($post_data as $key => $value): ?>
				<input type="hidden" name="<?php echo $key; ?>" value="<?php echo $value; ?>">
				<?php endforeach;?>
			</form>
			<?php echo \JText::_('JSHOP_REDIRECT_TO_PAYMENT_PAGE'); ?>
			<br/>
			<script type="text/javascript">document.getElementById('paymentform').submit();</script>
		</body>
		</html>
		<?php
		die();
	}

	function getUrlParams($pmconfigs) {
		$params = [];
		$params['order_id'] = \JFactory::getApplication()->input->getInt("reference");
		$params['hash'] = '';
		$params['checkHash'] = 0;
		$params['checkReturnParams'] = 1;
		return $params;
	}
	
	function getUrl() {
		return !empty($this->testmode) ? 'https://test.paygine.com' : 'https://pay.paygine.com';
	}
	
	/**
	 * @param $url string
	 * @param $data array
	 * @param $method string
	 * @return false|string
	 */
	function sendRequest($url, $data, $method = 'POST') {
		$query = http_build_query($data);
		$context = stream_context_create([
			'http' => [
				'header'  => "Content-Type: application/x-www-form-urlencoded\r\n"
					. "Content-Length: " . strlen($query) . "\r\n",
				'method'  => $method,
				'content' => $query
			]
		]);
		
		if (!$context)
			return false;
		
		return file_get_contents($url, false, $context);
	}
	
	/**
	 * @param array $data
	 * @param string $password
	 * @return void
	 */
	public function sign_data(array &$data, $password = '') {
		unset($data['signature']);
		$sign = implode('', $data);
		if($password)
			$sign .= $this->password;
		$data['signature'] = base64_encode(hash($this->hash_algo, $sign));
	}
	
	/**
	 * @param array|string $data
	 * @param string $password
	 * @return string
	 */
	public function generate_sign($data, $password = '') {
		$string = is_array($data) ? implode('', $data) : (string) $data;
		if($password)
			$string .= $this->password;
		return base64_encode(hash($this->hash_algo, $string));
	}
	
	public static function xml_array_to_string(array $array){
		$res = '';
		foreach($array as $key => $value) {
			if($key === '@attributes' or $key === 'signature') continue;
			$res .= is_string($value) ? $value : self::xml_array_to_string($value);
		}
		return $res;
	}
	
	static function centify($value){
		return intval(strval(round($value, 2) * 100));
	}
	
}